/*
as of March 05, 2010; no infinite intervals - as of March 16, 2010;
*/

#include <vector>
#include <string>
#include <stack>
#include <list>
#include <algorithm>

#include "mathlink.h"
#include "types.h"

namespace mlcxsc
{
    //*************************
    //helpers
    inline FunctionType GetFuncType(const std::string& aFuncName, int aArgCount)
    {
		FunctionType res;

		if(aFuncName == "Plus" && aArgCount > 0)
		{
			res = ftPlus;
		}
		else if(aFuncName == "Times" && aArgCount > 0)
		{
			res = ftTimes;
		}
		else if(aFuncName == "Power" && aArgCount == 2)
		{
			res = ftPower;
		}
		else if(aFuncName == "Sqrt" && aArgCount == 1)
		{
			res = ftSqrt;
		}
		else if(aFuncName == "Exp" && aArgCount == 1)
		{
			res = ftExp;
		}
		else if(aFuncName == "Log" && (aArgCount == 1 || aArgCount == 2))
		{
			res = ftLog;
		}
		else if(aFuncName == "Sin" && aArgCount == 1)
		{
			res = ftSin;
		}
		else if(aFuncName == "Cos" && aArgCount == 1)
		{
			res = ftCos;
		}
		else if(aFuncName == "Tan" && aArgCount == 1)
		{
			res = ftTan;
		}
		else if(aFuncName == "Cot" && aArgCount == 1)
		{
			res = ftCot;
		}
		else if(aFuncName == "Sinh" && aArgCount == 1)
		{
			res = ftSinh;
		}
		else if(aFuncName == "Cosh" && aArgCount == 1)
		{
			res = ftCosh;
		}
		else if(aFuncName == "Tanh" && aArgCount == 1)
		{
			res = ftTanh;
		}
		else if(aFuncName == "Coth" && aArgCount == 1)
		{
			res = ftCoth;
		}
		else if(aFuncName == "ArcSin" && aArgCount == 1)
		{
			res = ftArcSin;
		}
		else if(aFuncName == "ArcCos" && aArgCount == 1)
		{
			res = ftArcCos;
		}
		else if(aFuncName == "ArcTan" && aArgCount == 1)
		{
			res = ftArcTan;
		}
		else if(aFuncName == "ArcCot" && aArgCount == 1)
		{
			res = ftArcCot;
		}
		else if(aFuncName == "ArcSinh" && aArgCount == 1)
		{
			res = ftArcSinh;
		}
		else if(aFuncName == "ArcCosh" && aArgCount == 1)
		{
			res = ftArcCosh;
		}
		else if(aFuncName == "ArcTanh" && aArgCount == 1)
		{
			res = ftArcTanh;
		}
		else if(aFuncName == "ArcCoth" && aArgCount == 1)
		{
			res = ftArcCoth;
		}
		else if(aFuncName == "Csc" && aArgCount == 1)
		{
			res = ftCsc;
		}
		else if(aFuncName == "Sec" && aArgCount == 1)
		{
			res = ftSec;
		}
		else if((aFuncName == "Divide" || aFuncName == "div") && aArgCount == 2)
		{
			res = ftDivide;
		}
		else if(aFuncName == "Subtract" && aArgCount == 2)
		{
			res = ftSubtract;
		}
		else
		{
			res = ftNone;
		}

		return res;
    }
    
    /*
    ftSqrt
    ftExp
    ftLog
    trigonom
    
    aArg: the arg of the fun (may be an expression)
    */
	template<class Type>
    void InitUnary(FunctionType aFuncType, std::list<MLCXSCStackElementBase*>& aResult, const std::list<MLCXSCStackElementBase*>& aArg)
    {
		MLCXSCStackElementBase* arg = aArg.front();
		bool isconst = false;
		MLCXSCConst* val;
		if(arg->ElemType == etConst)
		{
			isconst = true;
			val = (MLCXSCConst*)arg;
		}

		MLCXSCFunction<Type>* pfunc;
		switch(aFuncType)
		{
			case ftLog:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncLogNat<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::ln(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftSqrt:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncSqrt<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::sqrt(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftExp:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncExp<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::exp(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftSin:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncSin<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::sin(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftCos:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncCos<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::cos(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftTan:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncTan<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::tan(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftCot:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncCot<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::cot(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftSinh:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncSinh<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::sinh(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftCosh:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncCosh<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::cosh(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftTanh:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncTanh<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::tanh(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftCoth:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncCoth<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::coth(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcSin:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcSin<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::asin(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcCos:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcCos<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::acos(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcTan:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcTan<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::atan(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcCot:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcCot<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::acot(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcSinh:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcSinh<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::asinh(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcCosh:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcCosh<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::acosh(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcTanh:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcTanh<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::atanh(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftArcCoth:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncArcCoth<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(cxsc::acoth(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftCsc:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncCsc<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(1.0/cxsc::sin(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			case ftSec:
				{
					if(!isconst)
					{
						pfunc = new MLCXSCFuncSec<Type>();
						aResult.push_back(pfunc);
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
					else
					{
						val->Set(1.0/cxsc::cos(val->IntervalValue));
						aResult.insert(aResult.end(), aArg.begin(), aArg.end());
					}
				}
				break;
			default:
				throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitUnary(): Illegal FuncType"));
		}
    }
    
    /*
    ftPower
    ftPlus
    ftTimes
	ftDivide
	ftSubtract
	ftLog
    */
	template <class Type>
    void InitBinary(FunctionType aFuncType, std::list<MLCXSCStackElementBase*>& aResult, const std::list<MLCXSCStackElementBase*>& aArg1, const std::list<MLCXSCStackElementBase*>& aArg2)
    {
		MLCXSCStackElementBase* arg1 = aArg1.front();
		MLCXSCStackElementBase* arg2 = aArg2.front();

		switch(aFuncType)
		{
			case ftPlus:
				{
					if(arg1->ElemType == etConst
						&& arg2->ElemType == etConst)
					{
						MLCXSCConst* a1 = (MLCXSCConst*)arg1;
						MLCXSCConst* a2 = (MLCXSCConst*)arg2;
						
						a1->Set(a1->IntervalValue + a2->IntervalValue);
						delete arg2;
						aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
					}
					else
					{
						MLCXSCFunction<Type>* pfunc;
						bool first = false; //whether arg1 should be included in aResult
						bool second = false; //whether arg2 should be included in aResult
						if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& arg2->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg2;
							pfunc = new MLCXSCFuncPlusConst<Type>(other->IntervalValue);
							first = true;
						}
						else if((arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction)
							&& arg1->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg1;
							pfunc = new MLCXSCFuncPlusConst<Type>(other->IntervalValue);
							second = true;
						}
						else if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& (arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction))
						{
							pfunc = new MLCXSCFuncPlus<Type>();
							first = second = true;
						}
						else
						{
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitBinary(): Illegal ElemType"));
						}

						aResult.push_back(pfunc);
						
						if(first)
						{
							aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
						}
						else
						{
							delete arg1;
						}
						if(second)
						{
							aResult.insert(aResult.end(), aArg2.begin(), aArg2.end());
						}
						else
						{
							delete arg2;
						}
					}
				}
				break;
			case ftSubtract:
				{
					if(arg1->ElemType == etConst
						&& arg2->ElemType == etConst)
					{
						MLCXSCConst* a1 = (MLCXSCConst*)arg1;
						MLCXSCConst* a2 = (MLCXSCConst*)arg2;
						
						a1->Set(a1->IntervalValue - a2->IntervalValue);
						delete arg2;
						aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
					}
					else
					{
						MLCXSCFunction<Type>* pfunc;
						bool first = false; //whether arg1 should be included in aResult
						bool second = false; //whether arg2 should be included in aResult
						if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& arg2->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg2;
							pfunc = new MLCXSCFuncSubtractMinuend<Type>(other->IntervalValue);
							first = true;
						}
						else if((arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction)
							&& arg1->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg1;
							pfunc = new MLCXSCFuncSubtractSubtrahend<Type>(other->IntervalValue);
							second = true;
						}
						else if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& (arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction))
						{
							pfunc = new MLCXSCFuncSubtract<Type>();
							first = second = true;
						}
						else
						{
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitBinary(): Illegal ElemType"));
						}

						aResult.push_back(pfunc);

						if(first)
						{
							aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
						}
						else
						{
							delete arg1;
						}
						if(second)
						{
							aResult.insert(aResult.end(), aArg2.begin(), aArg2.end());
						}
						else
						{
							delete arg2;
						}
					}
				}
				break;
			case ftLog:
				{
					if(arg1->ElemType == etConst
						&& arg2->ElemType == etConst)
					{
						MLCXSCConst* a1 = (MLCXSCConst*)arg1;
						MLCXSCConst* a2 = (MLCXSCConst*)arg2;
						
						if(a1->Special == scE)
						{
							a1->Set(cxsc::ln(a2->IntervalValue));
						}
						else
						{
							a1->Set(cxsc::ln(a2->IntervalValue)/cxsc::ln(a1->IntervalValue));
						}
						delete arg2;
						aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
					}
					else
					{
						MLCXSCFunction<Type>* pfunc;
						bool first = false; //whether arg1 should be included in aResult
						bool second = false; //whether arg2 should be included in aResult
						if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& arg2->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg2;
							pfunc = new MLCXSCFuncLogBase<Type>(other->IntervalValue);
							first = true;
						}
						else if((arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction)
							&& arg1->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg1;
							if(other->Special == scE)
							{
								pfunc = new MLCXSCFuncLogNat<Type>();
							}
							else
							{
								pfunc = new MLCXSCFuncLogFunc<Type>(other->IntervalValue);
							}
							second = true;
						}
						else if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& (arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction))
						{
							pfunc = new MLCXSCFuncLog<Type>();
							first = second = true;
						}
						else
						{
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitBinary(): Illegal ElemType"));
						}

						aResult.push_back(pfunc);
						
						if(first)
						{
							aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
						}
						else
						{
							delete arg1;
						}
						if(second)
						{
							aResult.insert(aResult.end(), aArg2.begin(), aArg2.end());
						}
						else
						{
							delete arg2;
						}
					}
				}
				break;
			case ftPower:
				{
					if(arg1->ElemType == etConst
						&& arg2->ElemType == etConst)
					{
						MLCXSCConst* a1 = (MLCXSCConst*)arg1;
						MLCXSCConst* a2 = (MLCXSCConst*)arg2;
						
						if(a1->Special == scE)
						{
							a1->Set(cxsc::exp(a2->IntervalValue));
						}
						else
						{
							a1->Set(cxsc::pow(a1->IntervalValue, a2->IntervalValue));
						}
						delete arg2;
						aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
					}
					else
					{
						MLCXSCFunction<Type>* pfunc;
						bool first = false;
						bool second = false;

						if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& arg2->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg2;
							
							if(other->IsInteger)
							{
								pfunc = new MLCXSCFuncPowerBaseInteger<Type>(other->IntegerValue);
							}
							else
							{
								pfunc = new MLCXSCFuncPowerBase<Type>(other->IntervalValue);
							}
							first = true;
						}
						else if((arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction)
							&& arg1->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg1;
							if(other->Special == scE)
							{
								pfunc = new MLCXSCFuncExp<Type>();
							}
							else
							{
								pfunc = new MLCXSCFuncPowerExp<Type>(other->IntervalValue);
							}
							second = true;
						}
						else if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& (arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction))
						{
							pfunc = new MLCXSCFuncPower<Type>();
							first = second = true;
						}
						else
						{
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitBinary(): Illegal ElemType"));
						}

						aResult.push_back(pfunc);

						if(first)
						{
							aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
						}
						else
						{
							delete arg1;
						}
						if(second)
						{
							aResult.insert(aResult.end(), aArg2.begin(), aArg2.end());
						}
						else
						{
							delete arg2;
						}
					}
				}
				break;
			case ftTimes:
				{
					if(arg1->ElemType == etConst
						&& arg2->ElemType == etConst)
					{
						MLCXSCConst* a1 = (MLCXSCConst*)arg1;
						MLCXSCConst* a2 = (MLCXSCConst*)arg2;
						
						a1->Set(a1->IntervalValue * a2->IntervalValue);
						delete arg2;
						aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
					}
					else
					{
						MLCXSCFunction<Type>* pfunc;
						bool first = false;
						bool second = false;
						if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& arg2->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg2;
							pfunc = new MLCXSCFuncTimesConst<Type>(other->IntervalValue);
							first = true;
						}
						else if((arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction)
							&& arg1->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg1;
							pfunc = new MLCXSCFuncTimesConst<Type>(other->IntervalValue);
							second = true;
						}
						else if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& (arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction))
						{
							pfunc = new MLCXSCFuncTimes<Type>();
							first = second = true;
						}
						else
						{
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitBinary(): Illegal ElemType"));
						}

						aResult.push_back(pfunc);

						if(first)
						{
							aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
						}
						else
						{
							delete arg1;
						}
						if(second)
						{
							aResult.insert(aResult.end(), aArg2.begin(), aArg2.end());
						}
						else
						{
							delete arg2;
						}
					}
				}
				break;
			case ftDivide:
				{
					if(arg1->ElemType == etConst
						&& arg2->ElemType == etConst)
					{
						MLCXSCConst* a1 = (MLCXSCConst*)arg1;
						MLCXSCConst* a2 = (MLCXSCConst*)arg2;
						
						a1->Set(a1->IntervalValue / a2->IntervalValue);
						delete arg2;
						aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
					}
					else
					{
						MLCXSCFunction<Type>* pfunc;
						bool first = false;
						bool second = false;
						if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& arg2->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg2;
							pfunc = new MLCXSCFuncDivideDividend<Type>(other->IntervalValue);
							first = true;
						}
						else if((arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction)
							&& arg1->ElemType == etConst)
						{
							MLCXSCConst* other = (MLCXSCConst*)arg1;
							pfunc = new MLCXSCFuncDivideDivisor<Type>(other->IntervalValue);
							second = true;
						}
						else if((arg1->ElemType == etVariable
							|| arg1->ElemType == etFunction)
							&& (arg2->ElemType == etVariable
							|| arg2->ElemType == etFunction))
						{
							pfunc = new MLCXSCFuncDivide<Type>();
							first = second = true;
						}
						else
						{
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitBinary(): Illegal ElemType"));
						}

						aResult.push_back(pfunc);

						if(first)
						{
							aResult.insert(aResult.end(), aArg1.begin(), aArg1.end());
						}
						else
						{
							delete arg1;
						}
						if(second)
						{
							aResult.insert(aResult.end(), aArg2.begin(), aArg2.end());
						}
						else
						{
							delete arg2;
						}
					}
				}
				break;
			default:
				throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("InitBinary(): Illegal FuncType"));
		}
    }
	
	template <class Type>
    void InitHelper(std::vector<MLCXSCVariable>& aVariables, std::list<MLCXSCStackElementBase*>& aResult)
    {
		switch(MLGetType(stdlink))
		{
			case MLTKINT:
				{
					int n;
					if(!MLGetInteger(stdlink, &n))
					{
/*						const char* msg = MLErrorMessage(stdlink);
						char* err_msg = new char[strlen(msg)+100];
						sprintf(err_msg, "InitHelper(): %s", msg);
						MLCXSCFunctionException e = MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, err_msg);
						delete err_msg;  */
MLCXSCFunctionException e = MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, MLErrorMessage(stdlink));
						MLClearError(stdlink);
						throw e;
					}
					aResult.push_back(new MLCXSCConst(n));
				}
				break;
			case MLTKREAL:
				{
					double n;
					if(!MLGetReal(stdlink, &n))
					{
/*						const char* msg = MLErrorMessage(stdlink);
						char* err_msg = new char[strlen(msg)+100];
						sprintf(err_msg, "InitHelper(): %s", msg);
						MLCXSCFunctionException e = MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, err_msg);
						delete err_msg;      */
MLCXSCFunctionException e = MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, MLErrorMessage(stdlink));
						MLClearError(stdlink);
						throw e;
					}
					aResult.push_back(new MLCXSCConst(cxsc::interval(n, n)));
				}
				break;
			case MLTKSYM: // E, Pi or variable
				{
					const char* name;
					MLGetSymbol(stdlink, &name);

					if(std::string("E") == name)
					{
						MLDisownSymbol(stdlink, name);
						aResult.push_back(new MLCXSCConst(cxsc::E_interval, scE));
					}
					else if(std::string("Pi") == name)
					{
						MLDisownSymbol(stdlink, name);
						aResult.push_back(new MLCXSCConst(cxsc::Pi_interval, scPi));
					}
					else
					{
						bool found = false;
						for(unsigned int i = 0; i < aVariables.size(); i++)
						{
							if(aVariables[i].Name == name)
							{
								aResult.push_back(&aVariables[i]);
								found = true;
								break;
							}
						}
						
						if(!found)
						{
							char* err_msg = new char[strlen(name)+100];
//							sprintf(err_msg, "Init Error: Unknown symbol %s", name);
							sprintf(err_msg, "Unknown variable %s", name);
							MLCXSCFunctionException e = MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string(err_msg));
							delete err_msg;
							MLDisownSymbol(stdlink, name);
							throw e;
						}
						
						MLDisownSymbol(stdlink, name);
					}
				}
				break;
			case MLTKFUNC: // Hold[] or Interval[{,}] (v intervala mozhe DirectedInfinity[+-1]) or one of the FunctionType-s or variable with an index; only Plus & Times can have more than 2 args
				{
					const char* name;
					long args;
					MLGetFunction(stdlink, &name, &args);

					if(std::string("Hold") == name)
					{
						MLDisownSymbol(stdlink, name);
						if(args != 1)
						{
							char err_msg[100];
							sprintf(err_msg, "Function Hold with %ld arguments", args);
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string(err_msg));
						}
						InitHelper<Type>(aVariables, aResult);
					}
					else if(std::string("Interval") == name)
					{
						MLDisownSymbol(stdlink, name);
						
						if(args == 1 && MLGetType(stdlink) == MLTKFUNC) // FUNC can be Hold or List
						{ 
					            long dim;
					            double* ibounds;
					            if(!MLGetRealList(stdlink, &ibounds, &dim))//  get interval bounds
					            {
					                char err_msg[100];
					                sprintf(err_msg, "interval bounds: %s", MLErrorMessage(stdlink));
					                MLClearError(stdlink);
	                throw mlcxsc::MLCXSCFunctionException(mlcxsc::MLCXSCFunctionException::ecExpressionError, err_msg);
					            }
					            aResult.push_back(new MLCXSCConst(cxsc::interval(ibounds[0], ibounds[1])));

						}
						else
						{
							char err_msg[100];
							sprintf(err_msg, "Interval must have a single argument List[.,.]");
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string(err_msg));
						}
					}
					else
					{
						FunctionType ftype = GetFuncType(name, args);
						if(ftype != ftNone) //function
						{
							switch(args)
							{
								case 1:
									{
										std::list<MLCXSCStackElementBase*> larg;
										InitHelper<Type>(aVariables, larg);
										InitUnary<Type>(ftype, aResult, larg);
									}
									break;
								case 2:
									{
										std::list<MLCXSCStackElementBase*> larg1;
										std::list<MLCXSCStackElementBase*> larg2;
										InitHelper<Type>(aVariables, larg1);
										InitHelper<Type>(aVariables, larg2);
										InitBinary<Type>(ftype, aResult, larg1, larg2);
									}
									break;
								default:
									{
										if(ftype == ftPlus || ftype == ftTimes)
										{
											std::list<MLCXSCStackElementBase*> curres;

											std::list<MLCXSCStackElementBase*> larg1;
											std::list<MLCXSCStackElementBase*> larg2;
											InitHelper<Type>(aVariables, larg1);
											InitHelper<Type>(aVariables, larg2);
											
											InitBinary<Type>(ftype, curres, larg1, larg2);
											for(int i = 0; i < args - 2; i++)
											{
												std::list<MLCXSCStackElementBase*> oldres = curres;
												
												curres.clear();
												
												std::list<MLCXSCStackElementBase*> larg;
												InitHelper<Type>(aVariables, larg);
												InitBinary<Type>(ftype, curres, oldres, larg);
											}

											aResult.insert(aResult.end(), curres.begin(), curres.end());
										}
										else // ne bi trqbvalo da se stiga do tuk za6oto GetFuncType() ve4e gleda i za broi parametri
										{
											throw MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string("Only +, * can have more than 2 args"));
										}
									}
									break;
							}
						}
						else //variable or unknown function
						{
							bool success = false;
							if(args == 1)
							{
								if(MLGetType(stdlink) == MLTKINT)
								{
									int n;
									MLGetInteger(stdlink, &n);
									
									char* var = new char[strlen(name)+(int)std::floor(std::log10((double)std::abs(n!=0?n:1)))+5];//1 za num of digits, 1 za null, 2 za [], 1 za -
									//char* var = new char[100];
									sprintf(var, "%s[%d]", name, n);
									
									for(unsigned int i = 0; i < aVariables.size(); i++)
									{
										if(aVariables[i].Name == var)
										{
											aResult.push_back(&aVariables[i]);
											success = true;
											break;
										}
									}
									delete var;
								}
							}
							
							if(!success)
							{
								char* err_msg = new char[strlen(name)+100];
								sprintf(err_msg, "Unknown function %s with %ld argument(s)", name, args);
								MLDisownSymbol(stdlink, name);
								MLCXSCFunctionException e = MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string(err_msg));
								delete err_msg;
								throw e;
							}
							MLDisownSymbol(stdlink, name);
						}
					}
				}
				break;
			default:
				throw MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string("Unknown Data Type"));
				break;
		}
    }
    
    //*************************
	//scalar function
	template <class Type>
    void MLCXSCFunctionScalar<Type>::Init(const std::string& aVariable)
    {
		std::vector<std::string> vars;
		vars.push_back(aVariable);
		Init(vars);
    }
	
	template <class Type>
    void MLCXSCFunctionScalar<Type>::Init(const std::vector<std::string>& aVariables)
    {
		mIsInitialized = false;
		
		if(!MLCXSCTypeTraits<Type>::IsDimensionValid(aVariables.size()))
		{
			char err_msg[100];
			sprintf(err_msg, "Init: Invalid dimension %ud of variables", aVariables.size());
			throw MLCXSCFunctionException(MLCXSCFunctionException::ecIllegalArgumentError, std::string(err_msg));
		}
		
		//init
		for(unsigned int i = 0; i < mDefinition.size(); i++)
		{
			switch(mDefinition[i]->ElemType)
			{
				case etFunction:
					{
						MLCXSCFunction<Type>* ptr = (MLCXSCFunction<Type>*)mDefinition[i];
						delete ptr;
					}
					break;
				case etVariable:
					//do nothing; it's the address of a member of mVariables
					break;
				case etConst: //mozhe da se slu4i ako f-qta e const
					{
						MLCXSCConst* ptr = (MLCXSCConst*)mDefinition[i];
						delete ptr;
					}
					break;
				default:
					throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("MLCXSCFunctionScalar<Type>::Init(): mDefinition contains illegal types"));
			}
		}
		
		mDefinition.clear();
		mVariables.clear();
		
		for(unsigned int i = 0; i < aVariables.size(); i++)
		{
			mVariables.push_back(MLCXSCVariable(aVariables[i], i+1));
		}

		//fill list
		std::list<MLCXSCStackElementBase*> res;
		InitHelper<Type>(mVariables, res);
		
		//definition
		mDefinition.insert(mDefinition.end(), res.begin(), res.end());
		
		mIsInitialized = true;
    }
	
	template <class Type>
    Type MLCXSCFunctionScalar<Type>::operator()(const Type& aVariableValue)
    {
		typename MLCXSCTypeTraits<Type>::VectorType vars = MLCXSCTypeTraits<Type>::GetVector(1);
		vars[1] = aVariableValue;
		return operator()(vars);
    }
	
	template <class Type>
    Type MLCXSCFunctionScalar<Type>::operator()(const typename MLCXSCTypeTraits<Type>::VectorType& aVariableValues)
    {
		if(!mIsInitialized)
		{
//			throw MLCXSCFunctionException(MLCXSCFunctionException::ecNotInitializedError, std::string("Calc Error: Init() should be called before calling operator()"));
			throw MLCXSCFunctionException(MLCXSCFunctionException::ecNotInitializedError, std::string("There is no initialized expression"));
		}
		
		unsigned int size = MLCXSCTypeTraits<Type>::GetVectorDim(aVariableValues);
		
		if(mVariables.size() != size)
		{
			char err_msg[100];
			sprintf(err_msg, "Calc: Illegal number of variables. Passed %d, must be %d", size, mVariables.size());
			throw MLCXSCFunctionException(MLCXSCFunctionException::ecIllegalArgumentError, std::string(err_msg));
		}

		std::vector<Type> calculation;

		for(int i = mDefinition.size() - 1; i >= 0; i--)
		{
			MLCXSCStackElementBase* pel = mDefinition[i];

			switch(pel->ElemType)
			{
				case etVariable:
					{
						int ind = ((MLCXSCVariable*)pel)->Number;
						calculation.push_back(aVariableValues[ind]);
					}
					break;
				case etFunction:
					{
						int argcnt = ((MLCXSCFunction<Type>*)pel)->ArgCount;
						std::vector<Type> args;
						for(int j = 0; j < argcnt; j++)
						{
							args.push_back(calculation.back());
							calculation.pop_back();
						}
						calculation.push_back((*((MLCXSCFunction<Type>*)pel))(args));
					}
					break;
				case etConst: //mozhe da se slu4i ako f-qta e const
					{
						calculation.push_back(MLCXSCTypeTraits<Type>::GetConst(((MLCXSCConst*)pel)->IntervalValue, mVariables.size()));
					}
					break;
				default:
					throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("MLCXSCFunctionScalar<Type>::operator(): Invalid FuncType"));
			}
		}

		if(calculation.size() != 1)
		{
			throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("MLCXSCFunctionScalar<Type>::operator(): Invalid calculated expression. Only one element should've been left"));
		}

		return calculation.front();
    }
	
	template <class Type>
    bool MLCXSCFunctionScalar<Type>::IsInitialized() const
	{
		return mIsInitialized;
	}
	
	template <class Type>
    void MLCXSCFunctionScalar<Type>::Invalidate()
	{
		mIsInitialized = false;
	}
	
	//*************************
	//multi-valued function
	template <class Type>
    void MLCXSCFunctionVector<Type>::Init(const std::string& aVariable)
    {
		std::vector<std::string> vars;
		vars.push_back(aVariable);
		Init(vars);
    }
	
	template <class Type>
    void MLCXSCFunctionVector<Type>::Init(const std::vector<std::string>& aVariables)
    {
		mIsInitialized = false;
		
		if(!MLCXSCTypeTraits<Type>::IsDimensionValid(aVariables.size()))
		{
			char err_msg[100];
			sprintf(err_msg, "Init: Invalid variables dimension of %ud", aVariables.size());
			throw MLCXSCFunctionException(MLCXSCFunctionException::ecIllegalArgumentError, std::string(err_msg));
		}
		
		switch(MLGetType(stdlink))
		{
			case MLTKFUNC:
				{
					const char* name;
					long args;
					MLGetFunction(stdlink, &name, &args);
					
					if(std::string("List") == name)
					{
						if((unsigned)args != aVariables.size())  
						{
							MLDisownSymbol(stdlink, name);
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecIllegalArgumentError, std::string("Init: The number of variables must be equal to the dimension of the function"));
						}
					}
					else
					{
						MLDisownSymbol(stdlink, name);
						throw MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string("Expression is not a list of functions"));
					}
					MLDisownSymbol(stdlink, name);
					break;
				}
			default:
				{
					throw MLCXSCFunctionException(MLCXSCFunctionException::ecExpressionError, std::string("Expression is not a list of functions"));
				}
				break;
		}
		
		//init
		for(unsigned int i = 0; i < mDefinition.size(); i++)
		{
			for(unsigned int j = 0; j < mDefinition[i].size(); j++)
			{
				switch(mDefinition[i][j]->ElemType)
				{
					case etFunction:
						{
							MLCXSCFunction<Type>* ptr = (MLCXSCFunction<Type>*)mDefinition[i][j];
							delete ptr;
						}
						break;
					case etVariable:
						//do nothing; it's the address of a member of mVariables
						break;
					case etConst: //mozhe da se slu4i ako f-qta e const
						{
							MLCXSCConst* ptr = (MLCXSCConst*)mDefinition[i][j];
							delete ptr;
						}
						break;
					default:
						throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("MLCXSCFunctionVector<Type>::Init(): mDefinition contains illegal types"));
				}
			}
		}
		
		mDefinition.clear();
		mVariables.clear();
		
		for(unsigned int i = 0; i < aVariables.size(); i++)
		{
			mVariables.push_back(MLCXSCVariable(aVariables[i], i+1));
		}

		//fill list
		for(unsigned int i = 0; i < mVariables.size(); i++)
		{
			std::vector<MLCXSCStackElementBase*> def;
			
			std::list<MLCXSCStackElementBase*> res;
			InitHelper<Type>(mVariables, res);
			
			def.insert(def.end(), res.begin(), res.end());
			mDefinition.push_back(def);
		}
		
		mIsInitialized = true;
    }
	
	template <class Type>
    typename MLCXSCTypeTraits<Type>::VectorType MLCXSCFunctionVector<Type>::operator()(const Type& aVariableValue)
    {
		typename MLCXSCTypeTraits<Type>::VectorType vars = MLCXSCTypeTraits<Type>::GetVector(1);
		vars[1] = aVariableValue;
		return operator()(vars);
    }
	
	template <class Type>
    typename MLCXSCTypeTraits<Type>::VectorType MLCXSCFunctionVector<Type>::operator()(const typename MLCXSCTypeTraits<Type>::VectorType& aVariableValues)
    {
		if(!mIsInitialized)
		{
//			throw MLCXSCFunctionException(MLCXSCFunctionException::ecNotInitializedError, std::string("Calc Error: Init() should be called before calling operator()"));
			throw MLCXSCFunctionException(MLCXSCFunctionException::ecNotInitializedError, std::string("There is no initialized expression"));
		}
		
		unsigned int size = MLCXSCTypeTraits<Type>::GetVectorDim(aVariableValues);
		
		if(mVariables.size() != size)
		{
			char err_msg[100];
			sprintf(err_msg, "Calc: Illegal number of variables. Passed %d, must be %d", size, mVariables.size());
			throw MLCXSCFunctionException(MLCXSCFunctionException::ecIllegalArgumentError, std::string(err_msg));
		}
		
		typename MLCXSCTypeTraits<Type>::VectorType res = MLCXSCTypeTraits<Type>::GetVector(mDefinition.size());
		for(unsigned int i = 0; i < mDefinition.size(); i++)
		{
			std::vector<Type> calculation;
			for(int j = mDefinition[i].size() - 1; j >= 0; j--)
			{
				MLCXSCStackElementBase* pel = mDefinition[i][j];

				switch(pel->ElemType)
				{
					case etVariable:
						{
							int ind = ((MLCXSCVariable*)pel)->Number;
							calculation.push_back(aVariableValues[ind]);
						}
						break;
					case etFunction:
						{
							int argcnt = ((MLCXSCFunction<Type>*)pel)->ArgCount;
							std::vector<Type> args;
							for(int k = 0; k < argcnt; k++)
							{
								args.push_back(calculation.back());
								calculation.pop_back();
							}
							calculation.push_back((*((MLCXSCFunction<Type>*)pel))(args));
						}
						break;
					case etConst: //mozhe da se slu4i ako f-qta e const
						{
							calculation.push_back(MLCXSCTypeTraits<Type>::GetConst(((MLCXSCConst*)pel)->IntervalValue, mVariables.size()));
						}
						break;
					default:
						throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("MLCXSCFunctionVector<Type>::operator(): Invalid FuncType"));
				}
			}
			
			if(calculation.size() != 1)
			{
				throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, std::string("MLCXSCFunctionVector<Type>::operator(): Invalid calculated expression. Only one element should've been left"));
			}
						
			res[i+1] = calculation.front();
		}

		return res;
    }
	
	template <class Type>
    bool MLCXSCFunctionVector<Type>::IsInitialized() const
	{
		return mIsInitialized;
	}
	
	template <class Type>
    void MLCXSCFunctionVector<Type>::Invalidate()
	{
		mIsInitialized = false;
	}
}
