/****************************
 implementation			as of March 3, 2010
				      March 8, 2010: see //EP
****************************/

#ifndef MLCXSC_STACKIMPL
#define MLCXSC_STACKIMPL

#include <vector>
#include <string>
#include <stdexcept>
#include <list>

#include "types.h"

namespace mlcxsc
{
	//*************************
	class MLCXSCFunctionException: public std::logic_error
	{
	public:
		enum ErrorCode
		{
			ecIllegalArgumentError, // Init() or operator() are called with illegal arguments
			ecNotInitializedError, 	// attempt to call operator() before successful execution of Init()
			ecExpressionError, 	// error in parsing the Mathematica expression in Init()
			ecInternalError 	// internal error that should not normally occur
		};
	private:
		ErrorCode mErrorCode;
	public:
		MLCXSCFunctionException(ErrorCode aErrorCode, const string& aMessage):std::logic_error(aMessage)
		{
			mErrorCode = aErrorCode;
		}
		
		ErrorCode GetErrorCode() const
		{
			return mErrorCode;
		}
	};

	//*************************
	template<class Type>
	class MLCXSCTypeTraits
	{
	public:
		typedef std::vector<Type> VectorType;
		
		static inline bool IsDimensionValid(int aDimension)
		{
			throw std::invalid_argument("Undefined type");
		}
		
		static inline Type GetConst(const cxsc::interval& aValue, int aDimension = 1)
		{
			throw std::invalid_argument("Undefined type");
		}
		
		static inline VectorType GetVector(int aDimension)
		{
			throw std::invalid_argument("Undefined type");
		}
		
		static inline int GetVectorDim(const VectorType& aVector)
		{
			throw std::invalid_argument("Undefined type");
		}
	};

	template<>
	class MLCXSCTypeTraits<DerivType>
	{
	public:
		typedef std::vector<DerivType> VectorType;
		
		static inline bool IsDimensionValid(int aDimension)
		{
			return aDimension==1 ; //EP  || aDimension==0
		}
		
		static inline DerivType GetConst(const cxsc::interval& aValue, int aDimension = 1)
		{
			if(IsDimensionValid(aDimension))
			{
				return DerivConst(aValue);
			}
			else
			{
				throw std::invalid_argument("Invalid dimension");
			}
		}
		
		static inline VectorType GetVector(int aDimension)
		{
			return std::vector<DerivType>(aDimension+1);
		}
		
		static inline int GetVectorDim(const VectorType& aVector)
		{
			return aVector.size() - 1;
		}
	};

	template<>
	class MLCXSCTypeTraits<GradType>
	{
	public:
		typedef GTvector VectorType;
		
		static inline bool IsDimensionValid(int aDimension)
		{
			return aDimension>=1; //EP aDimension>=0;
		}
		
		static inline GradType GetConst(const cxsc::interval& aValue, int aDimension = 1)
		{
			if(IsDimensionValid(aDimension))
			{
				GradType val(aDimension);
				val = aValue;
				return val;
			}
			else
			{
				throw std::invalid_argument("Invalid dimension");
			}
		}
		
		static inline VectorType GetVector(int aDimension)
		{
			return GTvector(aDimension);
		}
		
		static inline int GetVectorDim(const VectorType& aVector)
		{
			return aVector.Dim();
		}
	};

	template<>
	class MLCXSCTypeTraits<HessType>
	{
	public:
		typedef HTvector VectorType;
		
		static inline bool IsDimensionValid(int aDimension)
		{
			return aDimension>=1; //EP aDimension>=0;
		}
		
		static inline HessType GetConst(const cxsc::interval& aValue, int aDimension = 1)
		{
			if(IsDimensionValid(aDimension))
			{
				HessType val(aDimension);
				val = aValue;
				return val;
			}
			else
			{
				throw std::invalid_argument("Invalid dimension");
			}
		}
		
		static inline VectorType GetVector(int aDimension)
		{
			return HTvector(aDimension);
		}
		
		static inline int GetVectorDim(const VectorType& aVector)
		{
			return aVector.Dim();
		}
	};

	//*************************
	template<class Type>
	class MLCXSCFunctionScalar
	{
	protected:
		std::vector<MLCXSCStackElementBase*> mDefinition; //the definition
		std::vector<MLCXSCVariable> mVariables;
		
		bool mIsInitialized;
		
	public:
		MLCXSCFunctionScalar()
		{
			mIsInitialized = false;
		}
		~MLCXSCFunctionScalar()
		{
			for(unsigned int i = 0; i < mDefinition.size(); i++)
			{
				switch(mDefinition[i]->ElemType)
				{
					case etFunction:
						{
							MLCXSCFunction<Type>* ptr = (MLCXSCFunction<Type>*)mDefinition[i];
							delete ptr;
						}
						break;
					case etVariable:
						//do nothing; it's the address of a member of mVariables
						break;
					case etConst: //mozhe da se slu4i ako f-qta e const
						{
							MLCXSCConst* ptr = (MLCXSCConst*)mDefinition[i];
							delete ptr;
						}
						break;
					default:
						throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, "MLCXSCFunctionScalar::~MLCXSCFunctionScalar(): mDefinition contains illegal types");
				}
			}
		}
		MLCXSCFunctionScalar<Type>(const MLCXSCFunctionScalar<Type>& aOther)
		{
			throw std::runtime_error("Not Implemented");
		}
		MLCXSCFunctionScalar<Type>& operator=(const MLCXSCFunctionScalar<Type>& aOther)
		{
			throw std::runtime_error("Not Implemented");
		}
		
		void Init(const std::string& aVariable);
		void Init(const std::vector<std::string>& aVariables); //ne e definirano za DerivType ako length > 1
		
		Type operator()(const Type& aVariableValue);
		Type operator()(const typename MLCXSCTypeTraits<Type>::VectorType& aVariableValues); //ne e definirano za DerivType ako length > 1
		
		bool IsInitialized() const;
		void Invalidate();
	};

	//*************************
	//ograni4enie ot CXSC: vhodnata i izhodnata razmernost tr da sa ednakvi (t.e. trqbva mDefinition.size() == mVariables.size())
	//tova e za6oto razmernostite na GTvecor, HTvector sa gi napravili da ne mogat da bydat razli4ni ot razmernostite na GradType, HessType -ovete koito sydyrzhat
	template<class Type>
	class MLCXSCFunctionVector
	{
	protected:
		std::vector<std::vector<MLCXSCStackElementBase*> > mDefinition; //the definition (po 1 definition za vsqka razmernost)
		std::vector<MLCXSCVariable> mVariables;
		
		bool mIsInitialized;
		
	public:
		MLCXSCFunctionVector()
		{
			mIsInitialized = false;
		}
		~MLCXSCFunctionVector()
		{
			for(unsigned int i = 0; i < mDefinition.size(); i++)
			{
				for(unsigned int j = 0; j < mDefinition[i].size(); j++)
				{
					switch(mDefinition[i][j]->ElemType)
					{
						case etFunction:
							{
								MLCXSCFunction<Type>* ptr = (MLCXSCFunction<Type>*)mDefinition[i][j];
								delete ptr;
							}
							break;
						case etVariable:
							//do nothing; it's the address of a member of mVariables
							break;
						case etConst: //mozhe da se slu4i ako f-qta e const
							{
								MLCXSCConst* ptr = (MLCXSCConst*)mDefinition[i][j];
								delete ptr;
							}
							break;
						default:
							throw MLCXSCFunctionException(MLCXSCFunctionException::ecInternalError, "MLCXSCFunctionVector::~MLCXSCFunctionVector(): mDefinition contains illegal types");
					}
				}
			}
		}
		MLCXSCFunctionVector<Type>(const MLCXSCFunctionVector<Type>& aOther)
		{
			throw std::runtime_error("Not Implemented");
		}
		MLCXSCFunctionVector<Type>& operator=(const MLCXSCFunctionVector<Type>& aOther)
		{
			throw std::runtime_error("Not Implemented");
		}
		
		void Init(const std::string& aVariable);
		void Init(const std::vector<std::string>& aVariables); //ne e definirano za DerivType ako length > 1
		
		typename MLCXSCTypeTraits<Type>::VectorType operator()(const Type& aVariableValue);
		typename MLCXSCTypeTraits<Type>::VectorType operator()(const typename MLCXSCTypeTraits<Type>::VectorType& aVariableValues); //ne e definirano za DerivType ako length > 1
		
		bool IsInitialized() const;
		void Invalidate();
	};
}

//*************************
#include "expression.cpp"

#endif
