/****************************
Enums, types    as of January 26, 2010
****************************/

#ifndef MLCXSC_TYPES
#define MLCXSC_TYPES

#include <stdexcept>
#include <vector>
#include <string>

#include <ddf_ari.hpp>
#include <grad_ari.hpp>
#include <hess_ari.hpp>
#include <imath.hpp>

namespace mlcxsc
{
	enum StackElemType
	{
		etConst,
		etVariable,
		etFunction
	};

	enum FunctionType
	{
		ftNone,

		ftPlus,
		ftTimes,
		ftDivide,
		ftSubtract,

		ftPower,
		ftSqrt,
		ftExp,
		ftLog,
		ftSin,
		ftCos,
		ftTan,
		ftCot,
		ftSinh,
		ftCosh,
		ftTanh,
		ftCoth,
		ftArcSin,
		ftArcCos,
		ftArcTan,
		ftArcCot,
		ftArcSinh,
		ftArcCosh,
		ftArcTanh,
		ftArcCoth,

		ftCsc,
		ftSec  
	};
	
	enum SpecialConst
	{
		scNone,
		scE,
		scPi
	};

	struct MLCXSCStackElementBase
	{
		StackElemType ElemType;

		MLCXSCStackElementBase(StackElemType aElemType):ElemType(aElemType)
		{
		}
	};

	struct MLCXSCConst: public MLCXSCStackElementBase
	{
		cxsc::interval IntervalValue;
		int IntegerValue;
		bool IsInteger;
		SpecialConst Special;
		
		MLCXSCConst(const cxsc::interval& aIntervalValue, SpecialConst aSpecial = scNone):MLCXSCStackElementBase(etConst), IntervalValue(aIntervalValue), IntegerValue(0), IsInteger(false), Special(aSpecial)
		{
		}
		
		MLCXSCConst(int aIntegerValue, SpecialConst aSpecial = scNone):MLCXSCStackElementBase(etConst), IntervalValue(aIntegerValue, aIntegerValue), IntegerValue(aIntegerValue), IsInteger(true), Special(aSpecial)
		{
		}
		
		void Set(const cxsc::interval& aIntervalValue, SpecialConst aSpecial = scNone)
		{
			IntervalValue = aIntervalValue;
			IntegerValue = 0;
			IsInteger = false;
			Special = aSpecial;
		}
		
		void Set(int aIntegerValue, SpecialConst aSpecial = scNone)
		{
			IntervalValue = cxsc::interval(aIntegerValue, aIntegerValue);
			IntegerValue = aIntegerValue;
			IsInteger = true;
			Special = aSpecial;
		}
	};

	struct MLCXSCVariable: public MLCXSCStackElementBase
	{
		std::string Name;
		int Number;

		MLCXSCVariable(const std::string& aName, int aNumber):MLCXSCStackElementBase(etVariable), Name(aName), Number(aNumber)
		{
		}
	};

	template<class Type>
	struct MLCXSCFunction: public MLCXSCStackElementBase
	{
		FunctionType FuncType;
		int ArgCount;

		MLCXSCFunction(FunctionType aFuncType, int aArgCount):MLCXSCStackElementBase(etFunction), FuncType(aFuncType), ArgCount(aArgCount)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)=0;
	};

	//*** Functions ***
	template<class Type>
	struct MLCXSCFuncExp: public MLCXSCFunction<Type>
	{
		MLCXSCFuncExp():MLCXSCFunction<Type>(ftExp, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return exp(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncSqrt: public MLCXSCFunction<Type>
	{
		MLCXSCFuncSqrt():MLCXSCFunction<Type>(ftSqrt, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return sqrt(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncPowerBaseInteger: public MLCXSCFunction<Type>
	{
		int Exp;

		MLCXSCFuncPowerBaseInteger(int aExp):MLCXSCFunction<Type>(ftPower, 1), Exp(aExp)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			if(Exp == 2)
			{
				return sqr(aArgs[0]);
			}
			else
			{
				return power(aArgs[0], Exp);
			}
		}
	};

	template<class Type>
	struct MLCXSCFuncPowerBase: public MLCXSCFunction<Type>
	{
		cxsc::interval Exp;

		MLCXSCFuncPowerBase(const cxsc::interval& aExp):MLCXSCFunction<Type>(ftPower, 1), Exp(aExp)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return exp(ln(aArgs[0])*Exp);
		}
	};

	template<class Type>
	struct MLCXSCFuncPowerExp: public MLCXSCFunction<Type>
	{
		cxsc::interval Base;

		MLCXSCFuncPowerExp(const cxsc::interval& aBase):MLCXSCFunction<Type>(ftPower, 1), Base(aBase)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return exp(cxsc::ln(Base)*aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncPower: public MLCXSCFunction<Type>
	{
		MLCXSCFuncPower():MLCXSCFunction<Type>(ftPower, 2)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return exp(ln(aArgs[0])*aArgs[1]);
		}
	};

	template<class Type>
	struct MLCXSCFuncLogNat: public MLCXSCFunction<Type>
	{
		MLCXSCFuncLogNat():MLCXSCFunction<Type>(ftLog, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return ln(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncLogBase: public MLCXSCFunction<Type>
	{
		cxsc::interval Func;
		
		MLCXSCFuncLogBase(const cxsc::interval& aFunc):MLCXSCFunction<Type>(ftLog, 1), Func(aFunc)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return cxsc::ln(Func)/ ln(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncLogFunc: public MLCXSCFunction<Type>
	{
		cxsc::interval Base;
		
		MLCXSCFuncLogFunc(const cxsc::interval& aBase):MLCXSCFunction<Type>(ftLog, 1), Base(aBase)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return ln(aArgs[0])/cxsc::ln(Base);
		}
	};

	template<class Type>
	struct MLCXSCFuncLog: public MLCXSCFunction<Type>
	{
		MLCXSCFuncLog():MLCXSCFunction<Type>(ftLog, 2)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return ln(aArgs[1])/ ln(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncPlusConst: public MLCXSCFunction<Type>
	{
		cxsc::interval Other;

		MLCXSCFuncPlusConst(const cxsc::interval& aOther):MLCXSCFunction<Type>(ftPlus, 1), Other(aOther)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]+Other;
		}
	};

	template<class Type>
	struct MLCXSCFuncPlus: public MLCXSCFunction<Type>
	{
		MLCXSCFuncPlus():MLCXSCFunction<Type>(ftPlus, 2)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]+aArgs[1];
		}
	};

	template<class Type>
	struct MLCXSCFuncTimesConst: public MLCXSCFunction<Type>
	{
		cxsc::interval Other;

		MLCXSCFuncTimesConst(const cxsc::interval& aOther):MLCXSCFunction<Type>(ftTimes, 1), Other(aOther)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]*Other;
		}
	};

	template<class Type>
	struct MLCXSCFuncTimes: public MLCXSCFunction<Type>
	{
		MLCXSCFuncTimes():MLCXSCFunction<Type>(ftTimes, 2)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]*aArgs[1];
		}
	};

	template<class Type>
	struct MLCXSCFuncDivideDividend: public MLCXSCFunction<Type>
	{
		cxsc::interval Divisor;

		MLCXSCFuncDivideDividend(const cxsc::interval& aDivisor):MLCXSCFunction<Type>(ftDivide, 1), Divisor(aDivisor)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]/Divisor;
		}
	};

	template<class Type>
	struct MLCXSCFuncDivideDivisor: public MLCXSCFunction<Type>
	{
		cxsc::interval Dividend;

		MLCXSCFuncDivideDivisor(const cxsc::interval& aDividend):MLCXSCFunction<Type>(ftDivide, 1), Dividend(aDividend)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return Dividend/aArgs[0];
		}
	};

	template<class Type>
	struct MLCXSCFuncDivide: public MLCXSCFunction<Type>
	{
		MLCXSCFuncDivide():MLCXSCFunction<Type>(ftDivide, 2)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]/aArgs[1];
		}
	};

	template<class Type>
	struct MLCXSCFuncSubtractMinuend: public MLCXSCFunction<Type>
	{
		cxsc::interval Subtrahend;

		MLCXSCFuncSubtractMinuend(const cxsc::interval& aSubtrahend):MLCXSCFunction<Type>(ftSubtract, 1), Subtrahend(aSubtrahend)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]-Subtrahend;
		}
	};

	template<class Type>
	struct MLCXSCFuncSubtractSubtrahend: public MLCXSCFunction<Type>
	{
		cxsc::interval Minuend;

		MLCXSCFuncSubtractSubtrahend(const cxsc::interval& aMinuend):MLCXSCFunction<Type>(ftSubtract, 1), Minuend(aMinuend)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return Minuend-aArgs[0];
		}
	};

	template<class Type>
	struct MLCXSCFuncSubtract: public MLCXSCFunction<Type>
	{
		MLCXSCFuncSubtract():MLCXSCFunction<Type>(ftSubtract, 2)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return aArgs[0]-aArgs[1];
		}
	};

	template<class Type>
	struct MLCXSCFuncSin: public MLCXSCFunction<Type>
	{
		MLCXSCFuncSin():MLCXSCFunction<Type>(ftSin, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return sin(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncCos: public MLCXSCFunction<Type>
	{
		MLCXSCFuncCos():MLCXSCFunction<Type>(ftCos, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return cos(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncTan: public MLCXSCFunction<Type>
	{
		MLCXSCFuncTan():MLCXSCFunction<Type>(ftTan, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return tan(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncCot: public MLCXSCFunction<Type>
	{
		MLCXSCFuncCot():MLCXSCFunction<Type>(ftCot, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return cot(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncSinh: public MLCXSCFunction<Type>
	{
		MLCXSCFuncSinh():MLCXSCFunction<Type>(ftSinh, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return sinh(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncCosh: public MLCXSCFunction<Type>
	{
		MLCXSCFuncCosh():MLCXSCFunction<Type>(ftCosh, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return cosh(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncTanh: public MLCXSCFunction<Type>
	{
		MLCXSCFuncTanh():MLCXSCFunction<Type>(ftTanh, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return tanh(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncCoth: public MLCXSCFunction<Type>
	{
		MLCXSCFuncCoth():MLCXSCFunction<Type>(ftCoth, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return coth(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcSin: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcSin():MLCXSCFunction<Type>(ftArcSin, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return asin(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcCos: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcCos():MLCXSCFunction<Type>(ftArcCos, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return acos(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcTan: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcTan():MLCXSCFunction<Type>(ftArcTan, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return atan(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcCot: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcCot():MLCXSCFunction<Type>(ftArcCot, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return acot(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcSinh: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcSinh():MLCXSCFunction<Type>(ftArcSinh, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return asinh(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcCosh: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcCosh():MLCXSCFunction<Type>(ftArcCosh, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return acosh(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcTanh: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcTanh():MLCXSCFunction<Type>(ftArcTanh, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return atanh(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncArcCoth: public MLCXSCFunction<Type>
	{
		MLCXSCFuncArcCoth():MLCXSCFunction<Type>(ftArcCoth, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return acoth(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncCsc: public MLCXSCFunction<Type>
	{
		MLCXSCFuncCsc():MLCXSCFunction<Type>(ftCsc, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return 1.0/ sin(aArgs[0]);
		}
	};

	template<class Type>
	struct MLCXSCFuncSec: public MLCXSCFunction<Type>
	{
		MLCXSCFuncSec():MLCXSCFunction<Type>(ftSec, 1)
		{
		}

		virtual Type operator()(const std::vector<Type>& aArgs)
		{
			return 1.0/ cos(aArgs[0]);
		}
	};
}

#endif
