#include <vector>
#include <string>
#include <exception>
using namespace std;

#include "mathlink.h"
#include "ADExpressions/expression.h"

void PrintMLErrorMessage()
{
	char err_msg[100];
	sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorGrad::mlink,", MLErrorMessage(stdlink), "]");
	MLClearError(stdlink);
	MLNewPacket(stdlink);
	MLEvaluate(stdlink, err_msg);
	MLNextPacket(stdlink);
	MLNewPacket(stdlink);
	MLPutSymbol(stdlink, "$Failed");
}

void PrintErrorMessage(const mlcxsc::MLCXSCFunctionException& aException)
{
	char err_msg[1000];
	switch(aException.GetErrorCode())
	{
			case mlcxsc::MLCXSCFunctionException::ecIllegalArgumentError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorGrad::illargs,", aException.what(), "]");
				break;
			}
			case mlcxsc::MLCXSCFunctionException::ecNotInitializedError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorGrad::notrdy,", aException.what(), "]");
				break;
			}
			case mlcxsc::MLCXSCFunctionException::ecExpressionError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorGrad::expr,", aException.what(), "]");
				break;
			}
			case mlcxsc::MLCXSCFunctionException::ecInternalError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorGrad::internal,", aException.what(), "]");
				break;
			}
	}
	MLNewPacket(stdlink);
	MLEvaluate(stdlink, err_msg);
	MLNextPacket(stdlink);
	MLNewPacket(stdlink);
	MLPutSymbol(stdlink, "$Failed");
}

//************************ auxiliary functions *********************************

char* getVariable(void)
{
  char* var;
  switch(MLGetType(stdlink))
  {
    case MLTKSYM:
    {
      const char* name;
      MLGetSymbol(stdlink, &name);
      var = new char[strlen(name)+1];
      strcpy(var, name); 
      MLDisownSymbol(stdlink, name);
    }
    break;
    case MLTKFUNC:
    {  
      const char* name;
      long args;
      MLGetFunction(stdlink, &name, &args);
      bool success = false;
      if(args == 1)
      {
        if(MLGetType(stdlink) == MLTKINT)
        {
          int n;
          MLGetInteger(stdlink, &n);
          var = new char[strlen(name)+(int)std::floor(std::log10((double)std::abs(n!=0?n:1)))+5];
		 //  1 za num of digits, 1 za null, 2 za [], 1 za -
          sprintf(var, "%s[%d]", name, n);
          success = true;
          MLDisownSymbol(stdlink, name);
        }
      }
      if(!success)
      {
        MLDisownSymbol(stdlink, name);
        throw mlcxsc::MLCXSCFunctionException(mlcxsc::MLCXSCFunctionException::ecIllegalArgumentError, 
										std::string("Invalid variable name in the first argument"));
      }
    }
    break;
    default:
    {
      throw mlcxsc::MLCXSCFunctionException(mlcxsc::MLCXSCFunctionException::ecIllegalArgumentError,
										std::string("Invalid variable name in the first argument"));
    }
    break;
  }  
  return var;
}

//************************ function objects  ***********************************

mlcxsc::MLCXSCFunctionScalar<GradType> scalarfunc_grad;
mlcxsc::MLCXSCFunctionVector<GradType> vectorfunc_grad;

GradType sf_grad(const GTvector& x)
{
	return scalarfunc_grad(x);
}

GTvector vf_grad(const GTvector& x)
{
	return vectorfunc_grad(x);
}

//********************** initialization ****************************************

void SetFScalarGrad()
{ try
  {
     vector<string> vars;  // names of the variables

     const char* name;     // Mma function List
     long args;            // number of the arguments of List >= 1, both checked in Mma
     MLGetFunction(stdlink, &name, &args);
     MLDisownSymbol(stdlink, name);

     for(int i = 0; i < args; i++) vars.push_back(getVariable());
     scalarfunc_grad.Init(vars);  // defining the function
     MLPutSymbol(stdlink, "Null");
  }
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		scalarfunc_grad.Invalidate();
		PrintErrorMessage(e);
	}
}

void SetFVectorGrad()
{  try
  {
	vector<string> vars;  // names of the variables
	const char* name;     
	long args;      
     
// reading the variables
	MLGetFunction(stdlink, &name, &args); // name =List, args of List >=1 checked in Mma
	MLDisownSymbol(stdlink, name);
	for(int i = 0; i < args; i++) vars.push_back(getVariable());

// reading the functions		
	vectorfunc_grad.Init(vars);  // function definition
	MLPutSymbol(stdlink, "Null"); 
   }
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		vectorfunc_grad.Invalidate();
		PrintErrorMessage(e);
	}
}


//************************* evaluation *********************************************

void fValueScalarGrad()
{
  try
  {
    const char* name;
    long args, dim;
    MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
    MLDisownSymbol(stdlink, name);
     
    ivector vals(args);
    double* ibounds;
    for(int i = 1; i <= args; i++)
    {        
	if(!MLGetRealList(stdlink, &ibounds, &dim))
        {
          PrintMLErrorMessage();
          return;
        }
        vals[i] = interval(ibounds[0], ibounds[1]);
    }

    interval resf;
    GradType fx(args);
    fx = scalarfunc_grad(GradVar(vals));
    resf = fValue(fx); 

// Initializing C variables that will pass the computed results back to Mathematica
    double res_data[2];
    res_data[0] = _double(Inf(resf));
    res_data[1] = _double(Sup(resf));
    MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
  }
  catch(mlcxsc::MLCXSCFunctionException& e)
  {     PrintErrorMessage(e);	}
}


void gradValueScalar()
{
  try
  {
    const char* name; 
    long args, dim;
    MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
    MLDisownSymbol(stdlink, name);
    
    ivector vals(args);
    double* ibounds;
    for(int i = 1; i <= args; i++)   
    {
        if(!MLGetRealList(stdlink, &ibounds, &dim))
        {
          PrintMLErrorMessage(); 
          return;
        }
        vals[i] = interval(ibounds[0], ibounds[1]);
    }
 
    ivector resgrad;
    GradType fx(args);
    fx = scalarfunc_grad(GradVar(vals));
    resgrad = gradValue(fx);
    
// Initializing C variables that will pass the computed results back to Mathematica
    double res_data[2];
    MLPutFunction(stdlink, "List", VecLen(resgrad)); // grad:
    for(int i = 1; i <= VecLen(resgrad); i++)
    {
      res_data[0] = _double(Inf(resgrad[i]));
      res_data[1] = _double(Sup(resgrad[i]));
      MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
    }
  }  
  catch(mlcxsc::MLCXSCFunctionException& e)
  {     PrintErrorMessage(e);       }
}

void CalcScalarGrad()
{
	try
	{
		const char* name;
		long args, dim;
		MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
		MLDisownSymbol(stdlink, name);
		
		ivector vals(args);		
	        double* ibounds;
    		for(int i = 1; i <= args; i++)
    		{
        	  if(!MLGetRealList(stdlink, &ibounds, &dim))
        	  {
          		PrintMLErrorMessage();
          		return;
        	  }
        	  vals[i] = interval(ibounds[0], ibounds[1]);
    		}           

		interval resf;
		ivector resgrad;
		fgEvalG(sf_grad, vals, resf, resgrad);

// Initializing C variables that will pass the computed results back to Mathematica
		MLPutFunction(stdlink, "List", 2); // return f, gradf
		double res_data[2];  // f:
		res_data[0] = _double(Inf(resf));
		res_data[1] = _double(Sup(resf));
		MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
		
		MLPutFunction(stdlink, "List", VecLen(resgrad)); // grad:
		for(int i = 1; i <= VecLen(resgrad); i++)
		{
			res_data[0] = _double(Inf(resgrad[i]));
			res_data[1] = _double(Sup(resgrad[i]));
			MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
		}
	}
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		PrintErrorMessage(e);
	}
}

void fValueVectorGrad()
{
        try
        {
                const char* name;
                long args, dim;
                MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
                MLDisownSymbol(stdlink, name);
         
                ivector vals(args);
    		double* ibounds;
    		for(int i = 1; i <= args; i++)
    		{
        	  if(!MLGetRealList(stdlink, &ibounds, &dim))
        	  {
          		PrintMLErrorMessage();
          		return;
        	  }
        	  vals[i] = interval(ibounds[0], ibounds[1]);
    		}           

                ivector resf;
                GTvector fx(args);
                fx = vectorfunc_grad(GradVar(vals));
                resf = fValue(fx);
                                
                // Initializing C variables that will pass the computed results back to Mathematica
                double res_data[2];
                MLPutFunction(stdlink, "List", VecLen(resf));  // return f
                for(int i = 1; i <= VecLen(resf); i++)
                {
                        res_data[0] = _double(Inf(resf[i]));
                        res_data[1] = _double(Sup(resf[i]));
                        MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
                }
        }
        catch(mlcxsc::MLCXSCFunctionException& e)
        {
                PrintErrorMessage(e);
        }
}         

void JacValueGrad()
{
        try
        {
                const char* name;
                long args, dim;
                MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
                MLDisownSymbol(stdlink, name);
                
                ivector vals(args);
                double* ibounds;
                for(int i = 1; i <= args; i++)
                {
                  if(!MLGetRealList(stdlink, &ibounds, &dim))
                  {
                        PrintMLErrorMessage();
                        return;
                  }
                  vals[i] = interval(ibounds[0], ibounds[1]);
                }

                imatrix resgrad;
                GTvector fx(args);
                fx = vectorfunc_grad(GradVar(vals));
                resgrad = JacValue(fx);
                         
                // Initializing C variables that will pass the computed results back to Mathematica
		double res_data[2];
                MLPutFunction(stdlink, "List", ColLen(resgrad)); //gradf:
                for(int i = 1; i <= ColLen(resgrad); i++)
                {
                        imatrix_subv row = resgrad[i];
                        MLPutFunction(stdlink, "List", RowLen(resgrad));
                        for(int j = 1; j <= RowLen(resgrad); j++)
                        {
                                res_data[0] = _double(Inf(row[j]));
                                res_data[1] = _double(Sup(row[j]));
                                MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
                        }
                }
        }
        catch(mlcxsc::MLCXSCFunctionException& e)
        {
                PrintErrorMessage(e);
        }
}            

void CalcVectorGrad()
{
	try
	{
		const char* name;
		long args, dim;
		MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
		MLDisownSymbol(stdlink, name);
		
		ivector vals(args);
                double* ibounds;
                for(int i = 1; i <= args; i++)
                {
                  if(!MLGetRealList(stdlink, &ibounds, &dim))
                  {
                        PrintMLErrorMessage();
                        return;
                  }
                  vals[i] = interval(ibounds[0], ibounds[1]);
                }

                ivector resf;
                imatrix resgrad;
                fJEvalJ(vf_grad, vals, resf, resgrad);

// Initializing C variables that will pass the computed results back to Mathematica
		MLPutFunction(stdlink, "List", 2); // return resf, gradf
		//resf:
		double res_data[2];
		MLPutFunction(stdlink, "List", VecLen(resf));
		for(int i = 1; i <= VecLen(resf); i++)
		{
			res_data[0] = _double(Inf(resf[i]));
			res_data[1] = _double(Sup(resf[i]));
			MLPutRealList(stdlink, res_data, 2); // Send resf back to Mathematica
		}
		
		//gradf:
		MLPutFunction(stdlink, "List", ColLen(resgrad));
		for(int i = 1; i <= ColLen(resgrad); i++)
		{
			imatrix_subv row = resgrad[i];
			MLPutFunction(stdlink, "List", RowLen(resgrad));
			for(int j = 1; j <= RowLen(resgrad); j++)
			{
				res_data[0] = _double(Inf(row[j]));
				res_data[1] = _double(Sup(row[j]));
				MLPutRealList(stdlink, res_data, 2); // Send gradf back to Mathematica
			}
		}
	}
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		PrintErrorMessage(e);
	}
}


//****************** Function ReadyQ ******************************************************

void ReadyScalarGradQ()
{
	if(scalarfunc_grad.IsInitialized())
	{
		MLPutSymbol(stdlink, "True");
	}
	else
	{
		MLPutSymbol(stdlink, "False");
	}
}

void ReadyVectorGradQ()
{
	if(vectorfunc_grad.IsInitialized())
	{
		MLPutSymbol(stdlink, "True");
	}
	else
	{
		MLPutSymbol(stdlink, "False");
	}
}

//***************************** main *******************************************

int main(int argc, char* argv[]) // Standard MathLink main function
{
	return MLMain(argc, argv);
} 
