#include <vector>
#include <string>
#include <exception>
using namespace std;

#include "mathlink.h"
#include "ADExpressions/expression.h"

void PrintMLErrorMessage()
{
	char err_msg[100];
	sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorHess::mlink,", MLErrorMessage(stdlink), "]");
	MLClearError(stdlink);
	MLNewPacket(stdlink);
	MLEvaluate(stdlink, err_msg);
	MLNextPacket(stdlink);
	MLNewPacket(stdlink);
	MLPutSymbol(stdlink, "$Failed");
}

void PrintErrorMessage(const mlcxsc::MLCXSCFunctionException& aException)
{
	char err_msg[1000];
	switch(aException.GetErrorCode())
	{
			case mlcxsc::MLCXSCFunctionException::ecIllegalArgumentError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorHess::illargs,", aException.what(), "]");
				break;
			}
			case mlcxsc::MLCXSCFunctionException::ecNotInitializedError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorHess::notrdy,", aException.what(), "]");
				break;
			}
			case mlcxsc::MLCXSCFunctionException::ecExpressionError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorHess::expr,", aException.what(), "]");
				break;
			}
			case mlcxsc::MLCXSCFunctionException::ecInternalError:
			{
				sprintf(err_msg, "%s\"%.76s\"%s", "Message[MLCXSCErrorHess::internal,", aException.what(), "]");
				break;
			}
	}
	MLNewPacket(stdlink);
	MLEvaluate(stdlink, err_msg);
	MLNextPacket(stdlink);
	MLNewPacket(stdlink);
	MLPutSymbol(stdlink, "$Failed");
}

//************************ auxiliary functions *********************************

char* getVariable(void)
{
  char* var;
  switch(MLGetType(stdlink))
  {
    case MLTKSYM:
    {
      const char* name;
      MLGetSymbol(stdlink, &name);
      var = new char[strlen(name)+1];
      strcpy(var, name); 
      MLDisownSymbol(stdlink, name);
    }
    break;
    case MLTKFUNC:
    {  
      const char* name;
      long args;
      MLGetFunction(stdlink, &name, &args);
      bool success = false;
      if(args == 1)
      {
        if(MLGetType(stdlink) == MLTKINT)
        {
          int n;
          MLGetInteger(stdlink, &n);
          var = new char[strlen(name)+(int)std::floor(std::log10((double)std::abs(n!=0?n:1)))+5];
		 //  1 za num of digits, 1 za null, 2 za [], 1 za -
          sprintf(var, "%s[%d]", name, n);
          success = true;
          MLDisownSymbol(stdlink, name);
        }
      }
      if(!success)
      {
        MLDisownSymbol(stdlink, name);
        throw mlcxsc::MLCXSCFunctionException(mlcxsc::MLCXSCFunctionException::ecIllegalArgumentError, 
										std::string("Invalid variable name in the first argument"));
      }
    }
    break;
    default:
    {
      throw mlcxsc::MLCXSCFunctionException(mlcxsc::MLCXSCFunctionException::ecIllegalArgumentError,
										std::string("Invalid variable name in the first argument"));
    }
    break;
  }  
  return var;
}

//************************ function objects  ***********************************

mlcxsc::MLCXSCFunctionScalar<HessType> scalarfunc_hess;
mlcxsc::MLCXSCFunctionVector<HessType> vectorfunc_hess;

HessType sf_hess(const HTvector& x)
{          
        return scalarfunc_hess(x);  
}
        
HTvector vf_hess(const HTvector& x)
{                       
        return vectorfunc_hess(x); 
}

//********************** initialization ****************************************

void SetFScalarHess()
{ try
  {
     vector<string> vars;  // names of the variables
     const char* name;     
     long args;            
     MLGetFunction(stdlink, &name, &args); // name==List, args>=1, both checked in Mma
     MLDisownSymbol(stdlink, name);
     for(int i = 0; i < args; i++) vars.push_back(getVariable()); // reading the variable names
     scalarfunc_hess.Init(vars);  // reading the functional expression
     MLPutSymbol(stdlink, "Null");
  }
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		scalarfunc_hess.Invalidate();
		PrintErrorMessage(e);
	}
}

void SetFVectorHess()
{  try
  {
	vector<string> vars;  // names of the variables
	const char* name;     
	long args;      
	// reading the variables
	MLGetFunction(stdlink, &name, &args); // name =List, args of List >=1 checked in Mma
	MLDisownSymbol(stdlink, name);
	for(int i = 0; i < args; i++) vars.push_back(getVariable());

	vectorfunc_hess.Init(vars);  // reading the functional expressions
	MLPutSymbol(stdlink, "Null"); // sending the result back to Mma
   }
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		vectorfunc_hess.Invalidate();
		PrintErrorMessage(e);
	}
}


//************************* evaluation *********************************************

void fValueScalarHess()
{
  try
  {
    const char* name;
    long args, dim;
    MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
    MLDisownSymbol(stdlink, name);
     
    ivector vals(args);
    double* ibounds;
    for(int i = 1; i <= args; i++)
    {
      if(!MLGetRealList(stdlink, &ibounds, &dim))
      {
        PrintMLErrorMessage();
        return;
      }
      vals[i] = interval(ibounds[0], ibounds[1]);
    }
                
    interval resf;
    HessType fx(args);
    fx = scalarfunc_hess(HessVar(vals));
    resf = fValue(fx); 

// Initializing C variables that will pass the computed results back to Mathematica
    double res_data[2];
    res_data[0] = _double(Inf(resf));
    res_data[1] = _double(Sup(resf));
    MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
  }
  catch(mlcxsc::MLCXSCFunctionException& e)
  {     PrintErrorMessage(e);	}
}


void gradValueScalarHess()
{
  try
  {
    const char* name; 
    long args, dim;
    MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
    MLDisownSymbol(stdlink, name);
    
    ivector vals(args);
    double* ibounds;            
    for(int i = 1; i <= args; i++)            
    {            
      if(!MLGetRealList(stdlink, &ibounds, &dim))            
      {            
        PrintMLErrorMessage();                
        return;                
      }            
      vals[i] = interval(ibounds[0], ibounds[1]);            
    }          
 
    ivector resgrad;
    HessType fx(args);
    fx = scalarfunc_hess(HessVar(vals));
    resgrad = gradValue(fx);
    
// Initializing C variables that will pass the computed results back to Mathematica
    double res_data[2];
    MLPutFunction(stdlink, "List", VecLen(resgrad)); // grad:
    for(int i = 1; i <= VecLen(resgrad); i++)
    {
      res_data[0] = _double(Inf(resgrad[i]));
      res_data[1] = _double(Sup(resgrad[i]));
      MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
    }
  }  
  catch(mlcxsc::MLCXSCFunctionException& e)
  {     PrintErrorMessage(e);       }
}

void hessValueScalarHess()
{
  try
  {
    const char* name;
    long args, dim;
    MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
    MLDisownSymbol(stdlink, name);
    
    ivector vals(args);
    double* ibounds;   
    for(int i = 1; i <= args; i++)
    {
      if(!MLGetRealList(stdlink, &ibounds, &dim))
      {
        PrintMLErrorMessage();
        return;
      }
      vals[i] = interval(ibounds[0], ibounds[1]);
    }

    ivector reshess;
    HessType fx(args);
    fx = scalarfunc_hess(HessVar(vals));
    reshess = hessValue(fx);
   
// Initializing C variables that will pass the computed results back to Mathematica
    double res_data[2];
    MLPutFunction(stdlink, "List", VecLen(reshess)); // hess:
    for(int i = 1; i <= VecLen(reshess); i++)
    {
      res_data[0] = _double(Inf(reshess[i]));
      res_data[1] = _double(Sup(reshess[i]));
      MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
    }
  }
  catch(mlcxsc::MLCXSCFunctionException& e)
  {     PrintErrorMessage(e);       }
}

void fgEvalHess()
{
        try
        {
                const char* name;
                long args, dim;
                MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
                MLDisownSymbol(stdlink, name);
                
                ivector vals(args);
    		double* ibounds;   
    		for(int i = 1; i <= args; i++)
    		{
      		  if(!MLGetRealList(stdlink, &ibounds, &dim))
      		  {
        		PrintMLErrorMessage();
        		return;
      		  }
      		  vals[i] = interval(ibounds[0], ibounds[1]);
    		}

                interval resf;
                ivector resgrad;
                fgEvalH(sf_hess, vals, resf, resgrad);

// Initializing C variables that will pass the computed results back to Mathematica
                MLPutFunction(stdlink, "List", 2); // return f, gradf
                double res_data[2];  // f:
                res_data[0] = _double(Inf(resf));
                res_data[1] = _double(Sup(resf));
                MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
                        
                MLPutFunction(stdlink, "List", VecLen(resgrad)); // gradf:
                for(int i = 1; i <= VecLen(resgrad); i++)
                {
                        res_data[0] = _double(Inf(resgrad[i]));
                        res_data[1] = _double(Sup(resgrad[i]));
                        MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
		}
        }
        catch(mlcxsc::MLCXSCFunctionException& e)
        {
                PrintErrorMessage(e);
        }        
}

void CalcScalarHess()
{
	try
	{
		const char* name;
		long args, dim;
		MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
		MLDisownSymbol(stdlink, name);
		
		ivector vals(args);		
                double* ibounds;
                for(int i = 1; i <= args; i++)   
                {
                  if(!MLGetRealList(stdlink, &ibounds, &dim))
                  {
                        PrintMLErrorMessage();
                        return;
                  }
                  vals[i] = interval(ibounds[0], ibounds[1]);  
                }

		interval resf;
		ivector resgrad;
                imatrix reshess;
                fghEvalH(sf_hess, vals, resf, resgrad, reshess);

// Initializing C variables that will pass the computed results back to Mathematica
		MLPutFunction(stdlink, "List", 3); // return f, gradf, hessf
		double res_data[2];  // f:
		res_data[0] = _double(Inf(resf));
		res_data[1] = _double(Sup(resf));
		MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
		
		MLPutFunction(stdlink, "List", VecLen(resgrad)); // gradf:
		for(int i = 1; i <= VecLen(resgrad); i++)
		{
			res_data[0] = _double(Inf(resgrad[i]));
			res_data[1] = _double(Sup(resgrad[i]));
			MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
		}
                MLPutFunction(stdlink, "List", ColLen(reshess)); // hessf:
                for(int i = 1; i <= ColLen(reshess); i++)
                {
                        imatrix_subv row = reshess[i];
                        MLPutFunction(stdlink, "List", RowLen(reshess));
                        for(int j = 1; j <= RowLen(reshess); j++)
                        {
                                res_data[0] = _double(Inf(row[j]));
                                res_data[1] = _double(Sup(row[j]));
                                MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
                        }
                }

	}
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		PrintErrorMessage(e);
	}
}

void fValueVectorHess()
{
        try
        {
                const char* name;
                long args, dim;
                MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
                MLDisownSymbol(stdlink, name);
         
                ivector vals(args);
                double* ibounds;
                for(int i = 1; i <= args; i++)   
                {
                  if(!MLGetRealList(stdlink, &ibounds, &dim))
                  {
                        PrintMLErrorMessage();
                        return;
                  }
                  vals[i] = interval(ibounds[0], ibounds[1]);  
                }

                ivector resf;
                HTvector fx(args);
                fx = vectorfunc_hess(HessVar(vals));
                resf = fValue(fx);
                                
                // Initializing C variables that will pass the computed results back to Mathematica
                double res_data[2];
                MLPutFunction(stdlink, "List", VecLen(resf));  // return f
                for(int i = 1; i <= VecLen(resf); i++)
                {
                        res_data[0] = _double(Inf(resf[i]));
                        res_data[1] = _double(Sup(resf[i]));
                        MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
                }
        }
        catch(mlcxsc::MLCXSCFunctionException& e)
        {
                PrintErrorMessage(e);
        }
}         

void JacValueHess()
{
        try
        {
                const char* name;
                long args, dim;
                MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
                MLDisownSymbol(stdlink, name);
                
                ivector vals(args);
                double* ibounds;
                for(int i = 1; i <= args; i++)   
                {
                  if(!MLGetRealList(stdlink, &ibounds, &dim))
                  {
                        PrintMLErrorMessage();
                        return;
                  }
                  vals[i] = interval(ibounds[0], ibounds[1]);  
                }

                imatrix resgrad;
                HTvector fx(args);
                fx = vectorfunc_hess(HessVar(vals));
                resgrad = JacValue(fx);
                         
                // Initializing C variables that will pass the computed results back to Mathematica
		double res_data[2];
                MLPutFunction(stdlink, "List", ColLen(resgrad)); //gradf:
                for(int i = 1; i <= ColLen(resgrad); i++)
                {
                        imatrix_subv row = resgrad[i];
                        MLPutFunction(stdlink, "List", RowLen(resgrad));
                        for(int j = 1; j <= RowLen(resgrad); j++)
                        {
                                res_data[0] = _double(Inf(row[j]));
                                res_data[1] = _double(Sup(row[j]));
                                MLPutRealList(stdlink, res_data, 2); // Send the result back to Mathematica
                        }
                }
        }
        catch(mlcxsc::MLCXSCFunctionException& e)
        {
                PrintErrorMessage(e);
        }
}            

void CalcVectorHess()
{
	try
	{
		const char* name;
		long args, dim;
		MLGetFunction(stdlink, &name, &args); // name = List, args >=1, both checked in Mma
		MLDisownSymbol(stdlink, name);
		
		ivector vals(args);
                double* ibounds;
                for(int i = 1; i <= args; i++)   
                {
                  if(!MLGetRealList(stdlink, &ibounds, &dim))
                  {
                        PrintMLErrorMessage();
                        return;
                  }
                  vals[i] = interval(ibounds[0], ibounds[1]);  
                }

                ivector resf;
                imatrix resgrad;
                fJEvalJ(vf_hess, vals, resf, resgrad);

// Initializing C variables that will pass the computed results back to Mathematica
		MLPutFunction(stdlink, "List", 2); // return resf, gradf
		//resf:
		double res_data[2];
		MLPutFunction(stdlink, "List", VecLen(resf));
		for(int i = 1; i <= VecLen(resf); i++)
		{
			res_data[0] = _double(Inf(resf[i]));
			res_data[1] = _double(Sup(resf[i]));
			MLPutRealList(stdlink, res_data, 2); // Send resf back to Mathematica
		}
		
		//gradf:
		MLPutFunction(stdlink, "List", ColLen(resgrad));
		for(int i = 1; i <= ColLen(resgrad); i++)
		{
			imatrix_subv row = resgrad[i];
			MLPutFunction(stdlink, "List", RowLen(resgrad));
			for(int j = 1; j <= RowLen(resgrad); j++)
			{
				res_data[0] = _double(Inf(row[j]));
				res_data[1] = _double(Sup(row[j]));
				MLPutRealList(stdlink, res_data, 2); // Send gradf back to Mathematica
			}
		}
	}
	catch(mlcxsc::MLCXSCFunctionException& e)
	{
		PrintErrorMessage(e);
	}
}


//****************** Function ReadyQ ******************************************************

void ReadyScalarHessQ()
{
	if(scalarfunc_hess.IsInitialized())
		MLPutSymbol(stdlink, "True");
	else
		MLPutSymbol(stdlink, "False");
}

void ReadyVectorHessQ()
{
	if(vectorfunc_hess.IsInitialized())
		MLPutSymbol(stdlink, "True");
	else
		MLPutSymbol(stdlink, "False");
}

//***************************** main *******************************************

int main(int argc, char* argv[]) // Standard MathLink main function
{
	return MLMain(argc, argv);
} 
